function out=fn_F_val(LBF,rho)
%function to provide EOS quantities from a spline representation of
%Helmholtz energy as a function of a chosen strain metric.  In addition to the
%standard spline quantities, LBF needs (1) rho0 and (2) a strain metric added to the structure
% Usage: out=fn_F_val(LBF,rho)
% where     rho is density or inverse of whatever volume unit was used in
%               construction of the representation
%             out contains P, K and Kp=dKdP in the pressure units of the
%             construction
% JMB 2019
m=LBF.coefs(:);
k=LBF.order;
V=rho(:).^-1;
Vo=LBF.rho0.^-1;

if (isfield(LBF,'strainflg'))
 % strains need to be in increasing order 
if (strcmp(LBF.strainflg(1:3),'vol'))
   V=sort(V);  
else
   V=sort(V,'descend'); 
end
else
   error('The strain metric "LBF.strainflg" must be specified')
end


% determine strain and its derivatives wrt V
    out=getStrains(V,Vo,LBF.strainflg);
    f=out.f;
    fv=out.fv;
    f2v=out.f2v;
    f3v=out.f3v;

% get Helmholtz energy basis functions (that are functions of strain)
% The following gets the function and derivatives of the function spline
% basis functions at data site locations

Fcol=collocate(LBF.knots,k,f,k-1);

% Determine energy and  first three derivatives of energy wrt strain
F=Fcol(:,:,1)*m;
F1=Fcol(:,:,2)*m;
F2=Fcol(:,:,3)*m;
if k>3
 F3=Fcol(:,:,4)*m;
else
    F3=0;
end


% determine pressure and first two derivatives of P wrt volume dP/dV and d2P/dV2 
P=-fv.*F1(:);
P1=-F2(:).*fv.^2-F1(:).*f2v;  %derivative of previous line
P2=-F3(:).*fv.^3 -2*F2.*fv.*f2v -F2.*fv.*f2v - F1.*f3v; %derivative of previous line

% calculate energy, pressure, bulk modulus and its first derivative
out.rho=V.^-1;
out.V=V;
out.P=P;
out.F=F;
out.K=-V.*P1;
out.Kp=-(V.*P2./P1+1);

if (strcmp(LBF.strainflg(1:3),'vol'))
out.rho=out.rho(end:-1:1);
out.V=out.V(end:-1:1);
out.P=out.P(end:-1:1);
out.K=out.K(end:-1:1);
out.Kp=out.Kp(end:-1:1);
end

