<?php

date_default_timezone_set('Europe/Zagreb');

define('IMG_WIDTH', 640);
define('IMG_HEIGHT', 480);
define('PADDING_BOTTOM', 40);
define('PADDING_LEFT', 40);
define('VERTICAL_SCALE', 80);
define('HORIZONTAL_SCALE', 10);
//define('SPECTRAL_LENGTH', 200); // [Ma]
define('SPECTRAL_LENGTH', 48); // [Ma]
//define('SPECTRAL_SCALE_X', 3);
define('SPECTRAL_SCALE_X', 12.5);
define('SPECTRAL_SCALE_Y', 5);
define('USE_FOURIER_PP', true);	// Use phase and period (below), obtained by FA on the graph image
	$FA_PHASE = 9.355;	// phase (t0) [myr]
	$FA_PERIOD = 25.92; // period [myr]
define('OUT_DIR', 'OUT_GETEXT/');
define('OUTPUT_IMAGE', true);

// NOTE: keep number of decimals in dataset limited, otherwise unexpected results due to $resolution
// 14.5 or 11.6, 37.8 or 33.9, 113.1 or 117 ? Should be older boundary

$dataset = 4;

if ($dataset == 1) {	// From various sources on https://en.wikipedia.org/wiki/List_of_extinction_events and https://doi.org/10.1093/mnras/stv2088 ("Periodic impact cratering and extinction events over the last 260 million years" (2015), M. R. Rampino and Ken Caldeira)
	$extinctions_major = array(66, 201.3, 252.2, 365, 445);
	$extinctions_minor = array(37.8, 145, 260, 305, 420);
	$extinctions = array(11.6, 93.9, 182.7, 230, 270, 424, 428, 488, 502);
	$extinctions_potential = array(117, 168.3);
	$extinctions_potential_me = array();
	$FA_PHASE = 9.355;	// phase (t0)
	$FA_PERIOD = 25.92;
}
if ($dataset == 2) {	// From "A Concise Geologic Time Scale" (2016), F. M. Gradstein et al
	$extinctions_major = array(66, 201.4, 251.9, 372.2, 445.2);
	$extinctions_minor = array(37.8, 145, 259.8, 306.7, 419.2);
	$extinctions = array(11.6, 93.9, 183.7, 228.5, 272.3, 423, 427.4, 485.4, 500.5);
	$extinctions_potential = array(113.1, 168.3);
	$extinctions_potential_me = array();
	$FA_PHASE = 8.617;	// phase (t0)
	$FA_PERIOD = 26.00;
}
if ($dataset == 3) {	// From "A Concise Geologic Time Scale" (2016), F. M. Gradstein et al
	$extinctions_major = array(66, 201.4, 251.9, 372.2, 445.2);
	$extinctions_minor = array(37.8, 145, 259.8, 306.7, 419.2, 514);
	$extinctions = array(11.6, 93.9, 183.7, 228.5, 272.3, 423, 427.4, 485.4, 500.5, 541);
	$extinctions_potential = array(113.1, 168.3, 330.9);
	$extinctions_potential_me = array(295, 346.7, 393.3, 467.3);
	// dataset used for illustrative purposes, using default FA phase/period
}
if ($dataset == 4) {	// From "A Concise Geologic Time Scale" (2016), F. M. Gradstein et al
	$extinctions_major = array(66, 201.4, round((251.9 + 259.8) / 2, 1), 372.2, 445.2);
	$extinctions_minor = array(37.8, 145, 306.7, round((419.2 + 423) / 2, 1), round((514 + 541) / 2, 1));
	$extinctions = array(11.6, 93.9, round((183.7 + 168.3) / 2, 1), 228.5, round((272.3 + 295) / 2, 1), 427.4, round((485.4 + 467.3) / 2, 1), 500.5);
	$extinctions_potential = array(113.1, round((330.9 + 346.7) / 2, 1));
	$extinctions_potential_me = array(393.3);
	$FA_PHASE = 9.550;	// phase (t0)
	$FA_PERIOD = 25.89;
}
if ($dataset == 5) {	// From "A Concise Geologic Time Scale" (2016), F. M. Gradstein et al (like dataset 4 but with fused boundaries instead of a single boundary
	$extinctions_major = array(round((61.6 + 66) / 2, 1), round((199.4 + 201.4) / 2, 1), round((251.9 + 259.8) / 2, 1), round((372.2 + 382.7) / 2, 1), round((443.8 + 445.2) / 2, 1));
	$extinctions_minor = array(round((33.9 + 38) / 2, 1), round((139.4 + 145) / 2, 1), round((306.7 + 314.6) / 2, 1), round((419.2 + 423) / 2, 1), round((514 + 541) / 2, 1));
	$extinctions = array(round((11.6 + 13.8) / 2, 1), round((89.8 + 93.9) / 2, 1), round((183.7 + 168.3) / 2, 1), round((228.5 + 237) / 2, 1), round((272.3 + 295) / 2, 1), round((427.4 + 430.5) / 2, 1), round((485.4 + 467.3) / 2, 1), round((497 + 500.5) / 2, 1));
	$extinctions_potential = array(round((113.1 + 126.3) / 2, 1), round((330.9 + 346.7) / 2, 1));
	$extinctions_potential_me = array(round((387.7 + 393.3) / 2, 1));
	$FA_PHASE = 9.779;	// phase (t0)
	$FA_PERIOD = 25.84;
}
if ($dataset == 7) {	// all boundaries, From "A Concise Geologic Time Scale" (2016), F. M. Gradstein et al
	$extinctions_major = array(66.0, 201.4, 251.9, 372.2, 445.2);
	$extinctions_minor = array();
	$extinctions = array();
	$extinctions_potential = array();
	$extinctions_potential_me = array(0.004, 0.008, 0.012, 0.126, 0.773, 1.80, 2.58, 3.60, 5.33, 7.25, 11.63, 13.82, 15.97, 20.44, 23.03, 28.1, 33.9, 38.0, 41.0, 47.8, 56.0, 59.2, 61.6, 72.1, 84.2, 86.5, 89.8, 93.9, 100.5, 113.1, 126.3, 130.8, 134.7, 139.4, 145.0, 152.1, 157.3, 163.1, 166.1, 168.3, 170.3, 174.2, 183.7, 191.4, 199.4, 209.6, 228.5, 237.0, 241.5, 246.8, 249.8, 254.2, 259.8, 265.1, 268.8, 272.3, 282.0, 290.1, 295.0, 298.9, 303.4, 306.7, 314.6, 323.2, 330.9, 346.7, 358.9, 382.7, 387.7, 393.3, 407.6, 410.8, 419.2, 423.0, 425.6, 427.4, 430.5, 433.4, 438.5, 440.8, 443.8, 453.0, 458.4, 467.3, 470.0, 477.7, 485.4, 489.5, 494, 497, 500.5, 504.5, 509, 514, 520, 530, 541.0);
	$FA_PHASE = 8.617;	// phase (t0)
	$FA_PERIOD = 26.00;
}

$extinctions_pulses = array();

$pulse_phase = 66 - 61.985878446;

$impactors = array();

$real = array();
$real_minor = array();
$real_major = array();
$real_potential = array();
$real_potential_me = array();
$real_pulses = array();

$use_vedic = false;
$echo_all = false;
$echo_x = false;	// echo existence pulses

$e_avg = 1512000;
//$e = array(1728000, 1296000, 864000, 432000, 1728000);
$e = array(1728000, 1296000, 864000, 432000);
//$e = array(1728000, 1296000);
//$e = array(1296000, 1728000);
$e_cnt = count($e);

if ($use_vedic) {
	$Td = 25.92e6;
	$Td_offset = 14.16e6;
}
else {
	$Td = 25.704e6;
	$Td_offset = 14.592e6;
}
$Td_start = $Td - $Td_offset;

$parton_x = 24751.794;
$parton_d = 975014.206;

$resolution = 1000; // years
$echo_resolution = 100000;

function get_vedic_delay($year) {
	global $e, $e_cnt, $parton_x, $Td_start;
	$delay = 0; $i = 0; $cur_year = 0; $is_ex = false;
	while (1) {
		$e_cur = $e[$i % $e_cnt];
		$cur_year += $e_cur;
		if ($cur_year > $year) break;
		elseif ($cur_year == $year) $is_ex = true;
		if ($cur_year >= $Td_start)
			$delay += $parton_x;
		++$i;
	}
	return array($delay, $is_ex);
}

for ($i=-1, $time_delay=0, $e_current=0; $i<590000; $i++) {
	$do_echo = false; $is_x = false; $is_d = false; $out = '';
	$year = ($i+1)*$resolution;
	$year_ext = $year + $Td_start;
	if ($use_vedic) {
		$vedic_delay = get_vedic_delay($year_ext);
		$time_delay = $vedic_delay[0];
	}
	else {
		$time_delay = floor($year_ext / $e_avg) * $parton_x;
		$time_delay -= floor($Td_start / $e_avg) * $parton_x;
	}
	$time_delay += floor($year_ext / $Td) * $parton_d;
	$year_real = $year - $time_delay;
	$year_real_mil_unrounded = $year_real / 1000000;
	$year_real_mil = round($year_real_mil_unrounded, 3);
	$year_mil = $year / 1000000;
	if ($year % $echo_resolution == 0) $do_echo = true;
	if (($use_vedic && $vedic_delay[1]) || (!$use_vedic && ($year_ext % $e_avg == 0))) { $out .= '$ '; $is_x = true; $do_echo = true; } elseif ($do_echo) $out .= '  ';
	if ($year_ext % $Td == 0) { $out .= '# '; $is_d = true; $do_echo = true; $extinctions_pulses[] = $year_mil - $pulse_phase; $real_pulses[] = $year_real_mil; } elseif ($do_echo) $out .= '  ';
	if ($do_echo) {
		if (in_array($year_mil, $extinctions_major)) { $out .= '**** '; $real_major[] = $year_real_mil; }
		elseif (in_array($year_mil, $extinctions_minor)) { $out .= '***  '; $real_minor[] = $year_real_mil; }
		elseif (in_array($year_mil, $extinctions)) { $out .= '**   '; $real[] = $year_real_mil; }
		elseif (in_array($year_mil, $extinctions_potential)) { $out .= '*    '; $real_potential[] = $year_real_mil; }
		elseif (in_array($year_mil, $extinctions_potential_me)) { $out .= '*    '; $real_potential_me[] = $year_real_mil; }
		else { 
			if ($echo_all || ($echo_x && $is_x) || $is_d)
				$out .= '     ';
			else $do_echo = false;
		}
		$out .= $year_mil.' Ma | REAL: '.$year_real_mil_unrounded.' Ma'.PHP_EOL;
		if ($do_echo) echo $out;
	}
}

print_r($real_major);
print_r($real_minor);

if (!OUTPUT_IMAGE) exit;

$im = imagecreatetruecolor(IMG_WIDTH, IMG_HEIGHT);
$im_white = imagecolorallocate($im, 255, 255, 255);
$im_black = imagecolorallocate($im, 0, 0, 0);
$im_yellow = imagecolorallocate($im, 255, 255, 0);
$im_dark_yellow = imagecolorallocate($im, 128, 128, 0);
$im_medium_red = imagecolorallocate($im, 128, 0, 0);
$im_dark_blue = imagecolorallocate($im, 0, 0, 128);
$im_red = imagecolorallocate($im, 255, 0, 0);
$im_rust_fe = imagecolorallocate($im, 219, 59, 12);
$im_dark_red = imagecolorallocate($im, 128, 0, 0);
$im_green = imagecolorallocate($im, 0, 255, 0);
$im_dark_green = imagecolorallocate($im, 0, 128, 0);
$im_blue = imagecolorallocate($im, 0, 0, 255);
$im_dark_grey = imagecolorallocate($im, 64, 64, 64);
$im_grey = imagecolorallocate($im, 128, 128, 128);
imagefill($im, 0, 0, $im_white);

$img_name = 'extinctions'.($use_vedic?'_vedic':'');
$img_name = date('Y.m.d-H.i.s').'_'.$img_name;

function drawCoords($im, $color, $coord_lines=false, $coord_color=false) {
	imageline($im, PADDING_LEFT, 10, PADDING_LEFT, IMG_HEIGHT - 10, $color);								// vertical line
	imageline($im, 10, IMG_HEIGHT - PADDING_BOTTOM, IMG_WIDTH - 10, IMG_HEIGHT - PADDING_BOTTOM, $color);	// horizontal line
	// scales
	for ($i=1; $i<((IMG_HEIGHT - PADDING_BOTTOM) / VERTICAL_SCALE); $i++) {
		if ($coord_lines) imageline($im, PADDING_LEFT, (IMG_HEIGHT - PADDING_BOTTOM) - $i*VERTICAL_SCALE, IMG_WIDTH - 10, (IMG_HEIGHT - PADDING_BOTTOM) - $i*VERTICAL_SCALE, $coord_color);
		imageline($im, PADDING_LEFT - 3, (IMG_HEIGHT - PADDING_BOTTOM) - $i*VERTICAL_SCALE, PADDING_LEFT + 3, (IMG_HEIGHT - PADDING_BOTTOM) - $i*VERTICAL_SCALE, $color);
	}
	for ($i=1; $i<((IMG_WIDTH - PADDING_LEFT) / HORIZONTAL_SCALE); $i++) {
		if (($i%10 == 0) && $coord_lines) drawPulse($im, $i*HORIZONTAL_SCALE, $coord_color);
		imageline($im, PADDING_LEFT + $i*HORIZONTAL_SCALE, (IMG_HEIGHT - PADDING_BOTTOM) - ($i%10?3:7), PADDING_LEFT + $i*HORIZONTAL_SCALE, (IMG_HEIGHT - PADDING_BOTTOM) + ($i%10?3:7), $color);
	}
}

function drawPulse($im, $val, $color, $dashed=false) {
	if ($dashed)
		imagedashedline($im, PADDING_LEFT + $val, 10, PADDING_LEFT + $val, (IMG_HEIGHT - PADDING_BOTTOM), $color);
	else
		imageline($im, PADDING_LEFT + $val, 10, PADDING_LEFT + $val, (IMG_HEIGHT - PADDING_BOTTOM), $color);
}

function drawPixel($im, $x, $y, $color, $filled=true, $size = 7) {
	//imagesetpixel($im, PADDING_LEFT + $x, IMG_HEIGHT - ($y + PADDING_BOTTOM), $color);
	if ($filled)
		imagefilledellipse($im, PADDING_LEFT + $x, IMG_HEIGHT - ($y*VERTICAL_SCALE + PADDING_BOTTOM), $size, $size, $color);
	else
		imageellipse($im, PADDING_LEFT + $x, IMG_HEIGHT - ($y*VERTICAL_SCALE + PADDING_BOTTOM), $size, $size, $color);
}

drawCoords($im, $im_black);

$out_real_major = ''; $out_real_minor = ''; $out_real = ''; $out_real_potential = ''; $out_real_potential_me = ''; $out_extinctions_pulses = '';
$out_img_major = ''; $out_img_minor = ''; $out_img = ''; $out_img_potential = ''; $out_img_potential_me = '';
foreach ($real_major as $extinction) {
	drawPixel($im, $extinction, 5, $im_dark_red);
	$out_real_major .= ($out_real_major==''?'':', ').$extinction;
}
foreach ($extinctions_major as $extinction) {
	drawPixel($im, $extinction, 4.75, $im_dark_red, false);
	$out_img_major .= ($out_img_major==''?'':', ').$extinction;
}

foreach ($real_minor as $extinction) {
	drawPixel($im, $extinction, 4, $im_dark_yellow);
	$out_real_minor .= ($out_real_minor==''?'':', ').$extinction;
}
foreach ($extinctions_minor as $extinction) {
	drawPixel($im, $extinction, 3.75, $im_dark_yellow, false);
	$out_img_minor .= ($out_img_minor==''?'':', ').$extinction;
}

foreach ($real as $extinction) {
	drawPixel($im, $extinction, 3, $im_dark_green);
	$out_real .= ($out_real==''?'':', ').$extinction;
}
foreach ($extinctions as $extinction) {
	drawPixel($im, $extinction, 2.75, $im_dark_green, false);
	$out_img .= ($out_img==''?'':', ').$extinction;
}

foreach ($real_potential as $extinction) {
	drawPixel($im, $extinction, 2, $im_dark_blue);
	$out_real_potential .= ($out_real_potential==''?'':', ').$extinction;
}
foreach ($extinctions_potential as $extinction) {
	drawPixel($im, $extinction, 1.75, $im_dark_blue, false);
	$out_img_potential .= ($out_img_potential==''?'':', ').$extinction;
}

foreach ($real_potential_me as $extinction) {
	drawPixel($im, $extinction, 1, $im_rust_fe);
	$out_real_potential_me .= ($out_real_potential_me==''?'':', ').$extinction;
}
foreach ($extinctions_potential_me as $extinction) {
	drawPixel($im, $extinction, 0.75, $im_rust_fe, false);
	$out_img_potential_me .= ($out_img_potential_me==''?'':', ').$extinction;
}

foreach ($real_pulses as $extinction) {
	drawPulse($im, $extinction, $im_grey);
}
/*
$start = 66 - 2 * $Td / 1e6;
for ($i=$start; $i<590; $i+=($Td/1e6))
	drawPulse($im, $i, $im_grey);
*/
if (!USE_FOURIER_PP) {
	$PHASE_USED = 0; $PERIOD_USED = 0; $last_extinction = 0;
	foreach ($extinctions_pulses as $extinction) {
		if ($PHASE_USED == 0) $PHASE_USED = $extinction;
		drawPulse($im, $extinction, $im_grey, true);
		$out_extinctions_pulses .= ($out_extinctions_pulses==''?'':', ').$extinction;
		if ($last_extinction != 0 && $PERIOD_USED == 0) $PERIOD_USED = $extinction - $last_extinction;
		else
			$last_extinction = $extinction;
	}
}
else {
	// PERIOD FROM FOURIER ANALYSIS
	$PHASE_USED = $FA_PHASE;	// phase (t0)
	$PERIOD_USED = $FA_PERIOD;
	for ($i=$PHASE_USED; $i<590; $i+=$PERIOD_USED) {
		drawPulse($im, $i, $im_grey, true);
		$out_extinctions_pulses .= ($out_extinctions_pulses==''?'':', ').$i;
	}
}

imagepng($im, OUT_DIR.'('.$dataset.') '.$img_name.'.png');
imagedestroy($im);

echo 'Image saved to: '.OUT_DIR.'('.$dataset.') '.$img_name.'.png'.PHP_EOL;

echo 'Performing spectral analysis...';

$im = imagecreatetruecolor(IMG_WIDTH, IMG_HEIGHT);
$im_white = imagecolorallocate($im, 255, 255, 255);
$im_black = imagecolorallocate($im, 0, 0, 0);
$im_nice_blue = imagecolorallocatealpha($im, 22, 104, 236, 60);
$im_light_grey = imagecolorallocate($im, 192, 192, 192);
imagefill($im, 0, 0, $im_white);

$img_name = 'extinctions_csa'.($use_vedic?'_vedic':'');
$img_name = date('Y.m.d-H.i.s').'_'.$img_name;

drawCoords($im, $im_black, true, $im_light_grey);

// circular spectral analysis
$result = array(); $MAX_R = 0; $MAX_R_P = 0; $MAX_R_P_PHASE = 0;
for ($P=5; $P<SPECTRAL_LENGTH; $P += 0.001) {
	$N = 0; $S = 0; $C = 0;
	if (1) {
		foreach ($real_major as $extinction) {
			$a = sin((2*M_PI / $P) * $extinction);
			$b = cos((2*M_PI / $P) * $extinction);
			++$N;
			$S += $a;
			$C += $b;
		}
	}
	if (1) {
		foreach ($real_minor as $extinction) {
			$a = sin((2*M_PI / $P) * $extinction);
			$b = cos((2*M_PI / $P) * $extinction);
			++$N;
			$S += $a;
			$C += $b;
		}
	}
	if (1) {
		foreach ($real as $extinction) {
			$a = sin((2*M_PI / $P) * $extinction);
			$b = cos((2*M_PI / $P) * $extinction);
			++$N;
			$S += $a;
			$C += $b;
		}
	}
	if (1) {
		foreach ($real_potential as $extinction) {
			$a = sin((2*M_PI / $P) * $extinction);
			$b = cos((2*M_PI / $P) * $extinction);
			++$N;
			$S += $a;
			$C += $b;
		}
	}
	if (1) {
		foreach ($real_potential_me as $extinction) {
			$a = sin((2*M_PI / $P) * $extinction);
			$b = cos((2*M_PI / $P) * $extinction);
			++$N;
			$S += $a;
			$C += $b;
		}
	}
	$S /= $N;
	$C /= $N;
	$R = sqrt(pow($S, 2) + pow($C, 2));
	if ($R > $MAX_R) {
		$MAX_R = $R;
		$MAX_R_P = $P;
		$MAX_R_P_PHASE = ($P / (2*M_PI)) * atan($S / $C);
		if ($C < 0) $MAX_R_P_PHASE += $P / 2;
	}
	$result[] = array('P' => $P, 'R' => $R);
	//drawPixel($im, $x, $y, $color, $filled=true, $size = 7)
	drawPixel($im, $P * SPECTRAL_SCALE_X, $R * SPECTRAL_SCALE_Y, $im_nice_blue, true, 3);
}
echo ' DONE.'.PHP_EOL;

echo 'Selecting peaks...'.PHP_EOL;
function is_good_peak($result, $i, $result_cnt) {
	$MAX_LONGEVITY_MIN_RIGHT = 0.5 / 0.001;
	$MAX_LONGEVITY_MIN_LEFT = 0.25 / 0.001;
	for ($j=1; $j<$MAX_LONGEVITY_MIN_LEFT && ($i-$j)>-1; $j++)
		if ($result[$i-$j]['R'] > $result[$i]['R']) return false;
	for ($j=1; $j<$MAX_LONGEVITY_MIN_RIGHT && ($i+$j)<$result_cnt; $j++)
		if ($result[$i+$j]['R'] > $result[$i]['R']) return false;
	return true;
}
$result_cnt = count($result); $maximums = array(); $R_CUTOFF = 0.25;
for ($i=0; $i<$result_cnt; $i++) {
	if (($result[$i]['R'] >= $R_CUTOFF) && is_good_peak($result, $i, $result_cnt) && !in_array($result[$i], $maximums))
		$maximums[] = $result[$i];
	echo "\r".round((($i+1) / $result_cnt) * 100).'%';
}
echo "\rDONE.".PHP_EOL;

$img_name .= '_'.SPECTRAL_LENGTH;
imagepng($im, OUT_DIR.'('.$dataset.') '.$img_name.'.png');
imagedestroy($im);

echo 'Image saved to: '.OUT_DIR.'('.$dataset.') '.$img_name.'.png'.PHP_EOL;
$pre_out = PHP_EOL.'Fourier analysis'.PHP_EOL.'----------------'.PHP_EOL.'SPECTRAL_LENGTH: '.SPECTRAL_LENGTH.' Ma'.PHP_EOL.'SPECTRAL_SCALE_X: x'.SPECTRAL_SCALE_X.PHP_EOL.'SPECTRAL_SCALE_Y: x'.SPECTRAL_SCALE_Y.PHP_EOL;
$pre_out .= 'MAX_R = '.$MAX_R.PHP_EOL;
$pre_out .= 'MAX_R_P = '.$MAX_R_P.PHP_EOL;
$pre_out .= 'PHASE = '.$MAX_R_P_PHASE.' myr'.PHP_EOL;
$pre_out .= 'SELECTED PEAKS >= '.$R_CUTOFF.'R: ';
$out = '';
foreach ($maximums as $m)
	$out .= ($out==''?'':', ').'['.round($m['P'], 3).', '.$m['R'].']';
echo $pre_out.$out.PHP_EOL;
$pre_img = 'MAJOR EXTINCTIONS IMG AGES: '.$out_img_major.PHP_EOL.'MINOR EXTINCTIONS IMG AGES: '.$out_img_minor.PHP_EOL.'OTHER EXTINCTIONS IMG AGES: '.$out_img.PHP_EOL.'POTENTIAL EXTINCTIONS IMG AGES: '.$out_img_potential.PHP_EOL.'POTENTIAL (2) EXTINCTIONS IMG AGES: '.$out_img_potential_me.PHP_EOL;
$pre_out = $pre_img.PHP_EOL.'MAJOR EXTINCTIONS REAL AGES: '.$out_real_major.PHP_EOL.'MINOR EXTINCTIONS REAL AGES: '.$out_real_minor.PHP_EOL.'OTHER EXTINCTIONS REAL AGES: '.$out_real.PHP_EOL.'POTENTIAL EXTINCTIONS REAL AGES: '.$out_real_potential.PHP_EOL.'POTENTIAL (2) EXTINCTIONS REAL AGES: '.$out_real_potential_me.PHP_EOL.PHP_EOL.'EXTINCTION PULSES (GREY DASHED LINES): '.$out_extinctions_pulses.PHP_EOL.PHP_EOL.'PHASE USED: '.$PHASE_USED.' My'.PHP_EOL.'PERIOD USED: '.$PERIOD_USED.' My'.PHP_EOL.$pre_out;
file_put_contents(OUT_DIR.'('.$dataset.') '.$img_name.'.txt', $pre_out.$out.PHP_EOL);