% 水力计算函数hydcal
% 输入管径组合编码pipecom和风量、管长等info
% 输出不平衡管段数量unbal_num，不经济管段数量uneco_num，水力计算表data
function [unbal_num, uneco_num, Psummax, data] = hydcal(pipecode,info)
    if nargin < 2
        info = readpipe;
        if nargin < 1
            WandH = info(:,5:6);
            pipecode = encodepipe(cell2mat(WandH));
        end
    end
    unbal_num = []; uneco_num = []; Psummax = [];
    % 将输入的管径编号pipecode解码为矩形风管的宽和高WandH，无输入时使用原有设计的管径编号
    for current = 1:size(pipecode,1)
        currentpipe = pipecode(current,:);
        WandH = decodepipe(currentpipe);
        data = info;
        data(:, 5:7) = num2cell([WandH, currentpipe']);
        
        % 寻找每根风管的路径，从风管1（起点）经过哪些风管可到达
        connections = data(:, 2);
        for i = 1:length(connections)
            end_pipe = i;
            path = find_path(1, end_pipe, connections);
            data{i, 8} = path;
        end
        
        % 进行水力计算，输出不平衡管路数量obj和水力计算表data
        [unbal_i,uneco_i,Psummax_i,data] = cal(data);
        unbal_num = [unbal_num unbal_i];
        uneco_num = [uneco_num uneco_i];
        Psummax = [Psummax Psummax_i];
    end
    % 在data中添加列标题以便于阅读
    header = {'no', 'connections', 'cmh', 'length', 'width', 'height', ...
              'pipecode', 'path', 'velocity', 'lambda','deltaY', 'ksi', ...
              'deltaJ', 'deltaP', 'Psum', 'Pdown', 'Pmax', 'Pv', 'per', 'valve'};
    data = [header; data];
end


% 递归寻找路径path
function path = find_path(start_pipe, end_pipe, connections)
    path = [];
    if start_pipe == end_pipe
        path = [path, end_pipe];
    else
        for i = 1:length(connections{start_pipe})
            subpath = find_path(connections{start_pipe}(i), end_pipe, connections);
            if ~isempty(subpath)
                path = [start_pipe, subpath];
                break
            end
        end
    end
end

% 计算各管风速V，沿程阻力系数lambda，沿程阻力Y，局部阻力系数ksi，局部阻力J，总阻力P并写入data的第9~14列
function [unbal_num,uneco_num,Psummax,data] = cal(data)
    % 计算风速V
    c3 = cell2mat(data(:, 3));  % 第3列为风量 cmh
    c4 = cell2mat(data(:, 4));  % 第4列为管长 m
    c5 = cell2mat(data(:, 5));  % 第5列为宽   mm
    c6 = cell2mat(data(:, 6));  % 第6列为高   mm
    de = 2 * c5 .* c6 ./ (c5 + c6) / 1000;   % 计算当量直径 m 用于阻力计算
    V = c3 ./ (0.0036 * c5 .* c6);

    % 根据风速V是否在经济风速范围Vmin Vmax内，确定不经济管段数量，用于计算目标函数中的罚项
    Vmin = 1.5 * ones(size(V,1),1);
    Vmax = 3 * ones(size(V,1),1);
    mainpipe_position = find(cellfun('isempty', data(:,2)),1);
    Vmin(1:mainpipe_position) = 3;
    Vmax(1:mainpipe_position) = 5;
    uneco_indices = V < Vmin | V > Vmax;
    uneco_num = sum(uneco_indices);
    
    % 计算沿程阻力系数lambda和沿程阻力Y
    fun = @(x)1./x.^0.5+2*log10(0.00015/3.71./de+2.51./(V.*de./0.0000157)./x.^0.5);
    options = optimset('Display', 'off');
    lambda = fsolve(fun,0.02 * ones(length(V),1), options);
    Y = lambda ./ de .* (0.5 * 1.21 .* V .^ 2) .* c4;
    
    % 采用直接输入局部阻力系数ksi的方法计算局部阻力J
    % 由于目前仅有ksical.m可计算三通的局部阻力系数，其他变径、出入口等位置的局部阻力系数尚未录入
    % 因此直接输入设计工况下天正暖通自动计算的局部阻力系数进行计算
    ksi = [2.51	0 0	0.48 0 0 0 0 0 0 0 0.53 0 0	0 0 0.38 2.89 3.09 3.37 4.03 4.59 3.29 5.58 5.92 6.14 6.14 6.14 6.14 6.14 0.3 0.06 0.24 1.35 1.21 2.7 3.98 4.22 0.54 1.13 0 0 0 0 3.78 2.84 3.05 7.39 4.24 8.55 2.06 0 0 0 2.77 4.29 4.17]';
    J = ksi .* 0.5 * 1.21 .* V .^ 2;
    J(1) = J(1) + 130;
    
    % 计算总阻力P
    P = Y + J;
    
    % 写入data的第9~14列
    data(:, 9:14) = num2cell([V, lambda, Y, ksi, J, P]);
    [unbal_num,Psummax,data] = unbal(P,data);
end

% 计算不平衡管路数量 + 风速不经济管段数量
function [unbal_num,Psummax,data] = unbal(P,data)
    % 计算支路累加阻力Psum
    Psum = zeros;
    for i = 1:length(P)
        Psum(i) = sum(P(data{i, 8}));
    end

    % 计算后继总阻力Pdown
    Pdown = zeros;
    Psummax = max(Psum);
    Pdown(1) = Psummax;
    for i = 2:length(P)
        downstream = []; upstream = [];
        for j = 1:length(P)
            if any(data{j, 8} == i)
                downstream = [downstream j];
            elseif any(data{j, 2} == i)
                upstream = [upstream j];
            end
        end
        Pdown(i) = max(Psum(downstream)) - Psum(upstream);
        connections = data{i, 2};
        if isempty(connections)
            Pdown(i) = P(i);   
        end
    end

    % 计算并联最不利阻力Pmax
    Pmax = Pdown;   
    for i = 1:length(P)
        if length(data{i, 2}) > 1 
            for j = 1:length(data{i, 2})
                Pmax(data{i, 2}(j)) = max(Pdown(data{i, 2}));
            end
        end
    end
    
    % 计算平衡阀阻力Pv
    Pv = Pmax - Pdown;

    % 计算不平衡率，≥15%时为不平衡管路，计入unbal_num
    unbal_num = 0;
    per = Pv./Pmax;
    valve = zeros(1,length(P));
    for i = 1:length(P)
        if per(i) >= 0.15
            valve(i) = 1;
            unbal_num = unbal_num + 1;
        end
    end
    data(:, 15:20) = num2cell([Psum;Pdown;Pmax;Pv;per;valve]');

    % 将计算结果写入data 形成水力计算表
    data(:, 15:20) = num2cell([Psum;Pdown;Pmax;Pv;per;valve]'); 
end

